/*******************************************************************************
  ShapesDemo
  Example submitted by David Clegg

  This unit contains the Brush selection form.
*******************************************************************************/

using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.Drawing.Drawing2D;
using System.IO;

namespace ShapesDemo
{
	/// <summary>
	/// Summary description for WinForm.
	/// </summary>
	public class BrushForm : System.Windows.Forms.Form
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		private System.Windows.Forms.Button btnOK;
		private System.Windows.Forms.Button btnCancel;
		private System.Windows.Forms.Panel pnSolid;
		private System.Windows.Forms.Button btnSolidColour;
		private System.Windows.Forms.GroupBox gbHatch;
		private System.Windows.Forms.GroupBox gbLinearGradient;
		private System.Windows.Forms.GroupBox gbTexture;
		private System.Windows.Forms.Panel pnTexture;
		private System.Windows.Forms.Panel pnLinearGradient;
		private System.Windows.Forms.ComboBox cmbHatchStyles;
		private System.Windows.Forms.GroupBox gbSolid;
		private System.Windows.Forms.Button btnLinearStartColour;
		private System.Windows.Forms.Button btnLinearEndColour;

		//Default start and end colours for LinearGradientBrush
		private Color fLGStartColour = Color.Blue;
		private Color fLGEndColour = Color.Black;

		//Default start and end colours for HatchBrush
		private Color fHForeColour = Color.Black;
		private Color fHBackColour = Color.White;

		private Control fFocusedParent;
		private System.Drawing.Rectangle fFocusRectangle;
		private bool fFormShowing;

		private System.Windows.Forms.ColorDialog cd;
		private System.Windows.Forms.OpenFileDialog ofdImage;
		private System.Windows.Forms.Button btnImage;
		private System.Windows.Forms.Button btnHForeColour;
		private System.Windows.Forms.Button btnHBackColour;

		public BrushForm()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			//
			// TODO: Add any constructor code after InitializeComponent call
			//
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose (bool disposing)
		{
			if (disposing)
			{
				if (components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose(disposing);
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.btnOK = new System.Windows.Forms.Button();
			this.btnCancel = new System.Windows.Forms.Button();
			this.gbSolid = new System.Windows.Forms.GroupBox();
			this.btnSolidColour = new System.Windows.Forms.Button();
			this.pnSolid = new System.Windows.Forms.Panel();
			this.gbHatch = new System.Windows.Forms.GroupBox();
			this.btnHBackColour = new System.Windows.Forms.Button();
			this.btnHForeColour = new System.Windows.Forms.Button();
			this.cmbHatchStyles = new System.Windows.Forms.ComboBox();
			this.gbLinearGradient = new System.Windows.Forms.GroupBox();
			this.btnLinearEndColour = new System.Windows.Forms.Button();
			this.btnLinearStartColour = new System.Windows.Forms.Button();
			this.pnLinearGradient = new System.Windows.Forms.Panel();
			this.gbTexture = new System.Windows.Forms.GroupBox();
			this.btnImage = new System.Windows.Forms.Button();
			this.pnTexture = new System.Windows.Forms.Panel();
			this.cd = new System.Windows.Forms.ColorDialog();
			this.ofdImage = new System.Windows.Forms.OpenFileDialog();
			this.gbSolid.SuspendLayout();
			this.gbHatch.SuspendLayout();
			this.gbLinearGradient.SuspendLayout();
			this.gbTexture.SuspendLayout();
			this.SuspendLayout();
			// 
			// btnOK
			// 
			this.btnOK.DialogResult = System.Windows.Forms.DialogResult.OK;
			this.btnOK.Location = new System.Drawing.Point(224, 240);
			this.btnOK.Name = "btnOK";
			this.btnOK.TabIndex = 0;
			this.btnOK.Text = "OK";
			// 
			// btnCancel
			// 
			this.btnCancel.DialogResult = System.Windows.Forms.DialogResult.Cancel;
			this.btnCancel.Location = new System.Drawing.Point(312, 240);
			this.btnCancel.Name = "btnCancel";
			this.btnCancel.TabIndex = 1;
			this.btnCancel.Text = "Cancel";
			// 
			// gbSolid
			// 
			this.gbSolid.Controls.Add(this.btnSolidColour);
			this.gbSolid.Controls.Add(this.pnSolid);
			this.gbSolid.Location = new System.Drawing.Point(8, 8);
			this.gbSolid.Name = "gbSolid";
			this.gbSolid.Size = new System.Drawing.Size(184, 80);
			this.gbSolid.TabIndex = 6;
			this.gbSolid.TabStop = false;
			this.gbSolid.Text = "SolidBrush";
			this.gbSolid.Paint += new System.Windows.Forms.PaintEventHandler(this.BrushForm_Paint);
			// 
			// btnSolidColour
			// 
			this.btnSolidColour.Location = new System.Drawing.Point(72, 32);
			this.btnSolidColour.Name = "btnSolidColour";
			this.btnSolidColour.TabIndex = 6;
			this.btnSolidColour.Text = "Color...";
			this.btnSolidColour.Click += new System.EventHandler(this.btnSolidColour_Click);
			// 
			// pnSolid
			// 
			this.pnSolid.BackColor = System.Drawing.SystemColors.Control;
			this.pnSolid.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.pnSolid.Location = new System.Drawing.Point(16, 24);
			this.pnSolid.Name = "pnSolid";
			this.pnSolid.Size = new System.Drawing.Size(40, 40);
			this.pnSolid.TabIndex = 5;
			this.pnSolid.Click += new System.EventHandler(this.Panel_Click);
			this.pnSolid.Paint += new System.Windows.Forms.PaintEventHandler(this.pnSolid_Paint);
			// 
			// gbHatch
			// 
			this.gbHatch.BackColor = System.Drawing.SystemColors.Control;
			this.gbHatch.Controls.Add(this.btnHBackColour);
			this.gbHatch.Controls.Add(this.btnHForeColour);
			this.gbHatch.Controls.Add(this.cmbHatchStyles);
			this.gbHatch.ForeColor = System.Drawing.SystemColors.ControlText;
			this.gbHatch.Location = new System.Drawing.Point(200, 8);
			this.gbHatch.Name = "gbHatch";
			this.gbHatch.Size = new System.Drawing.Size(192, 168);
			this.gbHatch.TabIndex = 7;
			this.gbHatch.TabStop = false;
			this.gbHatch.Text = "HatchBrush";
			this.gbHatch.Paint += new System.Windows.Forms.PaintEventHandler(this.BrushForm_Paint);
			// 
			// btnHBackColour
			// 
			this.btnHBackColour.Location = new System.Drawing.Point(32, 120);
			this.btnHBackColour.Name = "btnHBackColour";
			this.btnHBackColour.Size = new System.Drawing.Size(120, 23);
			this.btnHBackColour.TabIndex = 9;
			this.btnHBackColour.Text = "Background Color...";
			this.btnHBackColour.Click += new System.EventHandler(this.btnHBackColour_Click);
			// 
			// btnHForeColour
			// 
			this.btnHForeColour.Location = new System.Drawing.Point(32, 80);
			this.btnHForeColour.Name = "btnHForeColour";
			this.btnHForeColour.Size = new System.Drawing.Size(120, 23);
			this.btnHForeColour.TabIndex = 8;
			this.btnHForeColour.Text = "Foreground Color...";
			this.btnHForeColour.Click += new System.EventHandler(this.btnHForeColour_Click);
			// 
			// cmbHatchStyles
			// 
			this.cmbHatchStyles.DrawMode = System.Windows.Forms.DrawMode.OwnerDrawFixed;
			this.cmbHatchStyles.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.cmbHatchStyles.ItemHeight = 18;
			this.cmbHatchStyles.Location = new System.Drawing.Point(8, 32);
			this.cmbHatchStyles.Name = "cmbHatchStyles";
			this.cmbHatchStyles.Size = new System.Drawing.Size(176, 24);
			this.cmbHatchStyles.Sorted = true;
			this.cmbHatchStyles.TabIndex = 7;
			this.cmbHatchStyles.Click += new System.EventHandler(this.cmbHatchStyles_SelectedIndexChanged);
			this.cmbHatchStyles.SelectedIndexChanged += new System.EventHandler(this.cmbHatchStyles_SelectedIndexChanged);
			this.cmbHatchStyles.DrawItem += new System.Windows.Forms.DrawItemEventHandler(this.cmbHatchStyles_DrawItem);
			// 
			// gbLinearGradient
			// 
			this.gbLinearGradient.Controls.Add(this.btnLinearEndColour);
			this.gbLinearGradient.Controls.Add(this.btnLinearStartColour);
			this.gbLinearGradient.Controls.Add(this.pnLinearGradient);
			this.gbLinearGradient.Location = new System.Drawing.Point(8, 184);
			this.gbLinearGradient.Name = "gbLinearGradient";
			this.gbLinearGradient.Size = new System.Drawing.Size(184, 80);
			this.gbLinearGradient.TabIndex = 8;
			this.gbLinearGradient.TabStop = false;
			this.gbLinearGradient.Text = "LinearGradiantBrush";
			this.gbLinearGradient.Paint += new System.Windows.Forms.PaintEventHandler(this.BrushForm_Paint);
			// 
			// btnLinearEndColour
			// 
			this.btnLinearEndColour.Location = new System.Drawing.Point(80, 45);
			this.btnLinearEndColour.Name = "btnLinearEndColour";
			this.btnLinearEndColour.Size = new System.Drawing.Size(83, 23);
			this.btnLinearEndColour.TabIndex = 9;
			this.btnLinearEndColour.Text = "End Color...";
			this.btnLinearEndColour.Click += new System.EventHandler(this.btnLinearEndColour_Click);
			// 
			// btnLinearStartColour
			// 
			this.btnLinearStartColour.Location = new System.Drawing.Point(80, 17);
			this.btnLinearStartColour.Name = "btnLinearStartColour";
			this.btnLinearStartColour.Size = new System.Drawing.Size(83, 23);
			this.btnLinearStartColour.TabIndex = 8;
			this.btnLinearStartColour.Text = "Start Color...";
			this.btnLinearStartColour.Click += new System.EventHandler(this.btnLinearStartColour_Click);
			// 
			// pnLinearGradient
			// 
			this.pnLinearGradient.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.pnLinearGradient.Location = new System.Drawing.Point(16, 24);
			this.pnLinearGradient.Name = "pnLinearGradient";
			this.pnLinearGradient.Size = new System.Drawing.Size(40, 40);
			this.pnLinearGradient.TabIndex = 7;
			this.pnLinearGradient.Click += new System.EventHandler(this.Panel_Click);
			this.pnLinearGradient.Paint += new System.Windows.Forms.PaintEventHandler(this.pnLinearGradient_Paint);
			// 
			// gbTexture
			// 
			this.gbTexture.Controls.Add(this.btnImage);
			this.gbTexture.Controls.Add(this.pnTexture);
			this.gbTexture.Location = new System.Drawing.Point(8, 96);
			this.gbTexture.Name = "gbTexture";
			this.gbTexture.Size = new System.Drawing.Size(184, 80);
			this.gbTexture.TabIndex = 10;
			this.gbTexture.TabStop = false;
			this.gbTexture.Text = "TextureBrush";
			this.gbTexture.Paint += new System.Windows.Forms.PaintEventHandler(this.BrushForm_Paint);
			// 
			// btnImage
			// 
			this.btnImage.Location = new System.Drawing.Point(72, 32);
			this.btnImage.Name = "btnImage";
			this.btnImage.TabIndex = 9;
			this.btnImage.Text = "Image...";
			this.btnImage.Click += new System.EventHandler(this.btnImage_Click);
			// 
			// pnTexture
			// 
			this.pnTexture.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.pnTexture.Location = new System.Drawing.Point(16, 24);
			this.pnTexture.Name = "pnTexture";
			this.pnTexture.Size = new System.Drawing.Size(40, 40);
			this.pnTexture.TabIndex = 6;
			this.pnTexture.Click += new System.EventHandler(this.Panel_Click);
			this.pnTexture.Paint += new System.Windows.Forms.PaintEventHandler(this.pnTexture_Paint);
			// 
			// ofdImage
			// 
			this.ofdImage.Filter = "Image files|*.bmp;*.jpg;*.jpeg;";
			// 
			// BrushForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(402, 276);
			this.Controls.Add(this.gbHatch);
			this.Controls.Add(this.gbSolid);
			this.Controls.Add(this.btnCancel);
			this.Controls.Add(this.btnOK);
			this.Controls.Add(this.gbTexture);
			this.Controls.Add(this.gbLinearGradient);
			this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedSingle;
			this.MaximizeBox = false;
			this.MinimizeBox = false;
			this.Name = "BrushForm";
			this.ShowInTaskbar = false;
			this.StartPosition = System.Windows.Forms.FormStartPosition.CenterScreen;
			this.Text = "Select Brush";
			this.Load += new System.EventHandler(this.BrushForm_Load);
			this.Paint += new System.Windows.Forms.PaintEventHandler(this.BrushForm_Paint);
			this.gbSolid.ResumeLayout(false);
			this.gbHatch.ResumeLayout(false);
			this.gbLinearGradient.ResumeLayout(false);
			this.gbTexture.ResumeLayout(false);
			this.ResumeLayout(false);
		}
		#endregion

		private static Brush fBrush;

		public static Brush Brush {
			get {return fBrush;}
			set {fBrush = value;}
		}

		/// <summary>
		/// Static method which can be used to show this form modally. It returns
		/// the DialogResult.
		/// </summary>
		public static DialogResult ShowBrushForm() {
			BrushForm lBrushForm = new BrushForm();
			return lBrushForm.ShowDialog();
		}

		private void BrushForm_Load(object sender, System.EventArgs e) {
			//Flag that the form is being shown, so certain events (e.g.
			//cmbHatch_SelectedIndexChanged) can perform conditional logic based
			//on this.
			fFormShowing = true;
			try {
				PopulateControls();
			}
			finally {
				fFormShowing = false;
			}
		}

		/// <summary>
		/// Populate the controls on the form and paint the panels.
		/// </summary>
		private void PopulateControls(){
			PopulateHatchStyles();
			SelectCurrentBrush();
		}

		/// <summary>
		/// Paints a Focus Rectangle around the currently selected brush.
		/// </summary>
		private void SelectCurrentBrush() {
			if (fBrush != null) {
				if (fBrush is SolidBrush)
					SelectBrush(pnSolid);
				else if (fBrush is TextureBrush)
					SelectBrush(pnTexture);
				else if (fBrush is LinearGradientBrush)
					SelectBrush(pnLinearGradient);
				else if (fBrush is HatchBrush)
					SelectHatchBrush();
			}
			else {
				//No brush currently selected, so default to a SolidBrush
				//with a colour of red
				fBrush = new SolidBrush(Color.Red);
				Panel_Click(pnSolid, null);
			}
		}

		/// <summary>
		/// Paints a Focus Rectangle around the selected panel, and ensures
		/// that the correct brush is associated with it.
		/// </summary>
		private void SelectBrush(Panel panel) {
			Panel_Click(panel, null);
		}

		/// <summary>
		/// Paints a Focus Rectangle around the Hatch Brush Combo Box and sets
		/// the item index to the currently selected brush.
		/// </summary>
		private void SelectHatchBrush() {
			fHForeColour = ((HatchBrush)fBrush).ForegroundColor;
			fHBackColour = ((HatchBrush)fBrush).BackgroundColor;
			DrawFocusRectangle(cmbHatchStyles);
			cmbHatchStyles.SelectedIndex = cmbHatchStyles.Items.IndexOf(((HatchBrush)fBrush).HatchStyle);
		}

		/// <summary>
		/// Populate the HatchStyles combo box.
		/// </summary>
		private void PopulateHatchStyles() {
			cmbHatchStyles.Items.Clear();

			foreach (HatchStyle hs in HatchStyle.GetValues(typeof(HatchStyle))) {
				//It initially appeared that a couple of HatchStyle values were
				//being duplicated (e.g. HatchStyle.Horizintal),
				//but further debugging showed that GetValues is indeed
				//returning duplicates. I will investigate further next time I'm
				//online...
				cmbHatchStyles.Items.Add(hs);
			}
			cmbHatchStyles.SelectedIndex = 0;
		}

		/// <summary>
		/// Fill the ClientRectangle area of the specified panel using the
		/// specified brush.
		/// </summary>
		private void PaintPanel(Panel panel, Brush brush) {
			Graphics lGraphics = panel.CreateGraphics();
			try {
				lGraphics.FillRectangle(brush, panel.ClientRectangle);
				//Save the brush so we can reference it if the panel is clicked
				panel.Tag = brush;
			}
			finally {
				//The Graphics reference must be disposed once we are finished
				//with it
				lGraphics.Dispose();
			}
		}

		/// <summary>
		/// Fill the ClientRectangle area of a Linear Gradient panel. For
		/// simplicity, this demo only demonstrates using a LinearGradientBrush
		/// created with a LinearGradientMode of Vertical
		/// </summary>
		private void PaintLinearGradientPanel() {
			LinearGradientBrush lGradientBrush =
				new LinearGradientBrush(pnLinearGradient.ClientRectangle, fLGStartColour,
										fLGEndColour, LinearGradientMode.Vertical);
			PaintPanel(pnLinearGradient, lGradientBrush);
		}

		private void PaintTexturePanel(string fileName) {
			if (File.Exists(fileName)) {
				Image image = Image.FromFile(fileName);
				TextureBrush textureBrush = new TextureBrush(image);
				PaintPanel(pnTexture, textureBrush);
			}
		}

		/// <summary>
		/// Draw each HatchStyle along with its name
		/// </summary>
		private void cmbHatchStyles_DrawItem(object sender, System.Windows.Forms.DrawItemEventArgs e) {
		   // Draw the background of the ListBox control for each item.
		   System.Drawing.Rectangle rect = new System.Drawing.Rectangle();
		   SolidBrush brush = new SolidBrush(Color.Black);
		   Pen pen = new Pen(Color.Black, 1);

		   rect.X = e.Bounds.X + 2;
		   rect.Y = e.Bounds.Y + 2;
		   rect.Height = e.Bounds.Height - 4;
		   rect.Width = e.Bounds.Height;

		   e.DrawBackground();
		   if (e.Index != -1) {

			   HatchStyle lStyle = (HatchStyle)cmbHatchStyles.Items[e.Index];
			   HatchBrush lBrush = new HatchBrush(lStyle, fHForeColour, fHBackColour);
			   e.Graphics.FillRectangle(lBrush, rect);
			   e.Graphics.DrawRectangle(pen, rect);
			   e.Graphics.DrawString(lStyle.ToString(), e.Font, brush, rect.X + rect.Width + 5, rect.Y + 2);
		   }
		   e.DrawFocusRectangle();
		}

		/// <summary>
		/// Show the colour dialog and change the colour of the Solid panel if
		/// appropriate.
		/// </summary>
		private void btnSolidColour_Click(object sender, System.EventArgs e) {
		  cd.Color = pnSolid.BackColor;
		  if (cd.ShowDialog() == DialogResult.OK) {
			pnSolid.BackColor = cd.Color;
			((SolidBrush)pnSolid.Tag).Color = cd.Color;
			//Ensure the Texture Panel has the selection rectangle around it,
			//and the current brush has been set
			Panel_Click(pnSolid, e);
		  }
		}

		/// <summary>
		/// Show the colour dialog to select the start colour of the
		/// LinearGradientBrush, and repaint the panel if appropriate.
		/// </summary>
		private void btnLinearStartColour_Click(object sender, System.EventArgs e) {
			if (cd.ShowDialog() == DialogResult.OK) {
				fLGStartColour = cd.Color;
				PaintLinearGradientPanel();
				//Ensure the focus rectangle is drawn around the Linear Gradient
				//panel
				SelectBrush(pnLinearGradient);
			}
		}

		/// <summary>
		/// Show the colour dialog to select the end colour of the
		/// LinearGradientBrush, and repaint the panel if appropriate.
		/// </summary>
		private void btnLinearEndColour_Click(object sender, System.EventArgs e) {
			if (cd.ShowDialog() == DialogResult.OK) {
				fLGEndColour = cd.Color;
				PaintLinearGradientPanel();
				//Ensure the focus rectangle is drawn around the Linear Gradient
				//panel
				SelectBrush(pnLinearGradient);
			}
		}

		/// <summary>
		/// Draw a focus rectangle around the TextureBrush panel, and fill the
		/// panel using the selected image.
		/// </summary>
		private void btnImage_Click(object sender, System.EventArgs e) {
			if (ofdImage.ShowDialog() == DialogResult.OK) {
				PaintTexturePanel(ofdImage.FileName);
				//Ensure the Texture Panel has the selection rectangle around it,
				//and the current brush has been set
				Panel_Click(pnTexture, e);
			}
		}

		/// <summary>
		/// Save the selected brush and draw the focus rectangle.
		/// </summary>
		private void Panel_Click(object sender, System.EventArgs e) {
			fBrush = (Brush)((Control)sender).Tag;
			DrawFocusRectangle(sender);
		}

		/// <summary>
		/// Draw a focus rectangle around the selected control.
		/// </summary>
		private void DrawFocusRectangle(object sender) {
			//Clear the previous focus rectangle
			ClearFocusRectangle();

			//Grab a reference to the control
			Control control = (Control)sender;

			//Create a rectangle 2 pixels either side of the controls dimensions.
			//Save a reference to the rectangle and the controls parent so
			//we can repaint it if necessary, and so we can clear the old focus
			//rectangle.
			fFocusRectangle =
				new System.Drawing.Rectangle(control.Left - 2, control.Top - 2,
											 control.Width + 4, control.Height + 4);
			fFocusedParent = control.Parent;

			//Grab a reference to the controls parents graphics area and draw
			//the focus rectangle.
			Graphics lGraphics = fFocusedParent.CreateGraphics();
			try {
				ControlPaint.DrawFocusRectangle(lGraphics, fFocusRectangle);
			}
			finally {
				//The Graphics reference must be disposed once we are finished
				//with it
				lGraphics.Dispose();
			}
		}

		/// <summary>
		/// Clears the previous focus rectangle.
		/// </summary>
		private void ClearFocusRectangle() {
			if (fFocusedParent != null) {
				Pen pen = new Pen(fFocusedParent.BackColor, 2);
				Graphics graphics = fFocusedParent.CreateGraphics();
				try {
					graphics.DrawRectangle(pen, fFocusRectangle);
				}
				finally {
					//The Graphics reference must be disposed once we are finished
					//with it
					graphics.Dispose();
				}
			}
		}

		/// <summary>
		/// Ensure that the focus rectangle is drawn if the form is repainted
		/// </summary>
		private void BrushForm_Paint(object sender, System.Windows.Forms.PaintEventArgs e) {
			//Redraw the focus rectangle
			if (fFocusedParent != null)
			{
				Graphics lGraphics = fFocusedParent.CreateGraphics();
				try {
					ControlPaint.DrawFocusRectangle(lGraphics,
													fFocusRectangle);
				}
				finally {
					//The Graphics reference must be disposed once we are finished
					//with it
					lGraphics.Dispose();
				}
			}
		}

		private void cmbHatchStyles_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			SetHatchBrush();
		}

		/// <summary>
		/// Create a new HatchBrush, based on the selected HatchStyle, and
		/// assign it to fBrush.
		/// </summary>
		private void SetHatchBrush() {
			if (fFormShowing == false) {
				if (fFocusedParent != gbHatch)
					DrawFocusRectangle(cmbHatchStyles);
				if (cmbHatchStyles.SelectedIndex != -1) {
					HatchStyle hs = (HatchStyle)cmbHatchStyles.Items[cmbHatchStyles.SelectedIndex];
					fBrush = new HatchBrush(hs, fHForeColour, fHBackColour);
				}
			}
		}

		private void pnTexture_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			if (fBrush is TextureBrush)
				PaintPanel(pnTexture, fBrush);
		}

		private void pnSolid_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			if (pnSolid.Tag != null)
				PaintPanel(pnSolid, (Brush)pnSolid.Tag);
			else
				PaintPanel(pnSolid, new SolidBrush(Color.Red));
		}

		private void pnLinearGradient_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			PaintLinearGradientPanel();
		}

		private void btnHForeColour_Click(object sender, System.EventArgs e)
		{
			if (cd.ShowDialog() == DialogResult.OK) {
				fHForeColour = cd.Color;
				HatchColourChanged();
			}
		}

		private void btnHBackColour_Click(object sender, System.EventArgs e)
		{
			if (cd.ShowDialog() == DialogResult.OK) {
				fHBackColour = cd.Color;
				HatchColourChanged();
			}
		}

		/// <summary>
		/// Create a new HatchBrush and assign it to fBrush, to ensure the new
		/// colours are used, and paint a focus rectangle around the Combo
		/// box.
		/// </summary>
		private void HatchColourChanged() {
			cmbHatchStyles.Refresh();
			SetHatchBrush();
			SelectHatchBrush();
		}
	}
}
